/*--------------------------------------------------------------------------*\

    FILE....: hda.c
    TYPE....: Linux device driver
    AUTHOR..: David Rowe
    DATE....: 22/12/01

    Linux kernel mode device driver for V12PCI (OpenSwitch12) and V6PCI cards.
    This is the main file of the kernel mode driver vpbhp.

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         V12PCI CT Card Software

         Copyright (C) David Rowe 2001 david@voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

			       DEFINES
							
\*---------------------------------------------------------------------------*/

#define	NAME	     "vpbhp"
#define SIZE_WD      0x100      // size of V12PCI card memory in bytes
#define MAX_V12PCI    4         // max number of V12PCI cards
#define SIZE_LCR     128        // size of 9052 local config reg space in bytes
#define SIZE_BUF     8096       // max size of V12PCI card block read/write
#define SIZE_FIFO    512        // size of FIFOs on prototype
#define SIZE_SW_FIFO 25600      // 100ms worth of samples

#define EEPROM_SIZE  64		// Eeprom size
#define EEPROM_CS    25		// control bits
#define EEPROM_CLK   24
#define EEPROM_RDBIT 27
#define EEPROM_WRBIT 26

#define PARTY			// Enable Party Conferencing
#define PARTY_MAX	10	// Maximum number of parties
#define PARTY_MAXMEM	10	// Maximum number of parties members

#define MODULE
#define __KERNEL__

/*---------------------------------------------------------------------------*\

			       INCLUDES
							
\*---------------------------------------------------------------------------*/

// Module version boilerplate
#include <linux/autoconf.h>
#if defined( CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#define MODVERSIONS
#endif

#ifdef MODVERSIONS
#include <linux/modversions.h>
#endif

#include <linux/ioport.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/fs.h>
#include <asm/segment.h>
#include <asm/uaccess.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <asm/system.h>
#include <linux/vmalloc.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <asm/uaccess.h>


#include "vpb_ioctl.h"
#include "fifoc.h"
#include "khook.h"
#include "kring.h"
#include "kldrop.h"


#if LINUX_VERSION_CODE > KERNEL_VERSION(2,4,13)
MODULE_LICENSE("GPL");
#endif

/*---------------------------------------------------------------------------*\

				GLOBALS
							
\*---------------------------------------------------------------------------*/

// ensure dynamic major number picked
unsigned int major = 0;

void isr(int irq, void *dev_id, struct pt_regs *regs);
/*
int ldropevt=0;
int new_ldropevt=0;
*/

static int vpbhp_ioctl(struct inode *, struct file *, unsigned int, 
		     unsigned long);

// alaw <-> linear stuff
#define ALAW_CODES 256
short linear2alaw(int sample);
int alaw2linear( short alawbyte);
void bitrev(unsigned char a[], int n) ;
static unsigned char    bitrev_lut[ALAW_CODES];

// eeprom access routines
void cntrlbit(int PCInn, int bit, int val);
void wrbit(int PCInn, int bit);
int rdbit(int PCInn);
unsigned int eeread(int PCIn, int addr);
void card_detail(int);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,13)
static struct file_operations vpbhp_fops={

	NULL,
	NULL,
	NULL, 
	NULL, 
	NULL,
	vpbhp_ioctl,
};
#else
static struct file_operations vpbhp_fops={

	owner: THIS_MODULE,
	llseek: NULL,
	read: NULL,
	write: NULL,
	poll: NULL,
	ioctl: vpbhp_ioctl,
	open: NULL,
	release: NULL,
	fasync: NULL
};
#endif

static int 	numPCI;   	// number of valid PCI devices detected
static	int 	HW_FRAME = 32;	// Size of data block from HW_FIFO
static int 	board_type=0;	// Set to 6 if Open Switch6

// translated base address of PLX9052 regions
static unsigned char  *base0[MAX_V12PCI]; // 9052 config registers (mem mapped)
static u32             base1[MAX_V12PCI]; // 9052 config registers (io mapped)
static unsigned short *base2[MAX_V12PCI]; // V12PCI card memory
static void           *khook[MAX_V12PCI]; // kernel mode hook detectors
static void           *kring[MAX_V12PCI]; // kernel mode ring detectors
static void           *kldrop[MAX_V12PCI]; // kernel mode ring detectors

// intermediate kernel space buffer from transfers between V12PCI memory 
// user memory (application)
static short 	buf[SIZE_BUF];

int 		irq[MAX_V12PCI];
void 		*tx_fifo[MAX_V12PCI];
void 		*rx_fifo[MAX_V12PCI];
static short 	default_tx[SIZE_FIFO/2];

// context info for each isr
int 		boardNum[MAX_V12PCI];

// buffer of samples to/from card
short 		framerx[MAX_V12PCI][SIZE_FIFO/2+1];
short 		frametx[MAX_V12PCI][SIZE_FIFO/2+1];

#ifdef PARTY
// Party Conferencing Information
struct party_guest {
	short board;
	short port;
};
struct party_invite {
	short party;
	short member;
};
static struct party_guest party_book[PARTY_MAX][PARTY_MAXMEM];
static struct party_invite party_people[3][12];
static short parties[PARTY_MAX];
short party_count;
short sbridge=0;
#endif
#define BRIDGING
#ifdef BRIDGING
struct bridge {
	short board;
	short port;
};
struct bridge london[10][12];
int sbridge_count=0;
#endif


/*---------------------------------------------------------------------------*\

				   FUNCTIONS
							
\*---------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: init_module()
	AUTHOR...: David Rowe
	DATE.....: 22/12/01

	Called by the O/S when the module is loaded. Searches for
	valid V12PCI cards.

\*--------------------------------------------------------------------------*/

int init_module(void) {
	char 	s[4];
	struct 	pci_dev *dev = NULL;
	u32 	ret, oe, intctrl, i, waitstates;
	short 	cntrl, x,y;
	char  	pciipr, my_irq;

	major = register_chrdev(major, NAME, &vpbhp_fops);
//	printk("<1>vpbhp: major = %d\n",major);
	printk(KERN_INFO NAME": Registered Major = %d\n",major);

	// search for V12PCI devices on PCI bus.  Note we are actually using
	// a PLX9052, but this is functionally equiv to the 9050.

	numPCI = 0;
	while((dev = pci_find_device(PCI_VENDOR_ID_PLX, PCI_DEVICE_ID_PLX_9050,
				     dev)) != NULL) {

		// check that subsytem ID & Subsytem Vendor matches
		pci_read_config_dword(dev, 0x2c, (u32*)s);
 		//printk("s[3] = %c 0x%02x s[2] = %c 0x%02x\n",s[3],s[3]&0xff,
		//       s[2],s[2]&0xff);
		
		// Do if "12VT" or " 6VT" // representing OpenSwitch 6 or 12
		if (((s[3] == '1')&&(s[2] == '2')&&(s[1] == 'V')&&(s[0] == 'T')) |
		   ((s[3] == ' ')&&(s[2] == '6')&&(s[1] == 'V')&&(s[0] == 'T')))
	       	{
      
		    // OK, V12PCI or V6PCI found, so map address regions.....

		    if((s[3] == '1')&&(s[2] == '2'))	
		         printk(NAME": Found V12PCI Card %d !\n", numPCI);
		    
		    if((s[3] == ' ')&&(s[2] == '6'))	
		    {
			//HW_FRAME=6;    
			board_type=6;		// change loop ranges for kloop etc.
						// only good for single card type!!!
		        printk(NAME": Found V6PCI Card %d !\n", numPCI);
		    }

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,13)
			base0[numPCI] = ioremap(dev->base_address[0] &
						PCI_BASE_ADDRESS_MEM_MASK,
						SIZE_LCR);
			base2[numPCI] = ioremap(dev->base_address[2] &
						PCI_BASE_ADDRESS_MEM_MASK,
						sizeof(short)*SIZE_WD);

			// make sure no-one access's LCR via I/O space mapping
			base1[numPCI] = dev->base_address[1] & 
			                PCI_BASE_ADDRESS_IO_MASK;
			ret = check_region(base1[numPCI], SIZE_LCR);
			request_region(base1[numPCI], SIZE_LCR, NAME);
#else
                        base0[numPCI] = ioremap(dev->resource[0].start &
                                                PCI_BASE_ADDRESS_MEM_MASK,
                                                SIZE_LCR);
                        base2[numPCI] = ioremap(dev->resource[2].start &
                                                PCI_BASE_ADDRESS_MEM_MASK,
                                                sizeof(short)*SIZE_WD);
 
                        // make sure no-one access's LCR via I/O space mapping
                        base1[numPCI] = dev->resource[1].start &
                                        PCI_BASE_ADDRESS_IO_MASK;
                        ret = check_region(base1[numPCI], SIZE_LCR);
                        request_region(base1[numPCI], SIZE_LCR, NAME);
#endif
			waitstates = readl(base0[numPCI]+0x28);
			//printk("waitstates[%d] = 0x%x\n", numPCI, waitstates);
			writel(0x40410020, base0[numPCI]+0x28);     
			
			// set up OE generator

			// test value for development
			oe = 0x00000005;
			//oe = 0xffffffff;
			//oe = 0x00001000;
			x = oe & 0xff;
			memcpy_toio(base2[numPCI]+2, &x, sizeof(short));
			x = (oe>>8) & 0xff;
			memcpy_toio(base2[numPCI]+3, &x, sizeof(short));
			x = (oe>>16) & 0xff;
			memcpy_toio(base2[numPCI]+4, &x, sizeof(short));
			x = (oe>>24) & 0xff;
			memcpy_toio(base2[numPCI]+5, &x, sizeof(short));    
			
			// reset HW (FIFOs and codecs)
			cntrl = 0x00;
			memcpy_toio(base2[numPCI]+6, &cntrl, sizeof(short));
			cntrl = 0x80;
			memcpy_toio(base2[numPCI]+6, &cntrl, sizeof(short));
			cntrl = 0x00;
			memcpy_toio(base2[numPCI]+6, &cntrl, sizeof(short));
			
                        // create software FIFOs

			fifo_create(&tx_fifo[numPCI], SIZE_SW_FIFO*sizeof(short));
			fifo_create(&rx_fifo[numPCI], SIZE_SW_FIFO*sizeof(short));

			// set up default TX frame
			for(i=0; i<32; i++) {
				default_tx[i] = 0;
			}

			// test values for development
			default_tx[0] = 0x55;
			default_tx[31] = 0x01;

			// create rest of default frame
			for(i=32; i<SIZE_FIFO/2; i++) {
				default_tx[i] = default_tx[i-32];
			}

			// create hook detector
			khook[numPCI] = khook_open(base2[numPCI]);

			// create ring detector
			kring[numPCI] = kring_open(base2[numPCI]);

			// create Loop Drop detector
			kldrop[numPCI] = kldrop_open(base2[numPCI]);

			// set up interrupts (DO THIS LAST!!)
			// one int per card  any number of cards

			ret = pci_read_config_byte(dev, PCI_INTERRUPT_LINE, 
						   &my_irq);
			irq[numPCI] = my_irq;
			printk(KERN_INFO NAME": Board[%d] irq = %d\n",numPCI,(int)irq[i]);
			ret = pci_read_config_byte(dev, 0x3d, &pciipr);     
			//printk("pciipr = 0x%x\n",(int)pciipr);
			boardNum[numPCI] = numPCI;
			if (irq != 0) {
				ret = request_irq(irq[numPCI], isr, 
						  SA_INTERRUPT, 
						  NAME, &boardNum[numPCI]);
				if (ret) {
					// note: this version of driver cant 
					// share ints
					printk(KERN_WARNING NAME": can't assign irq!\n");
					irq[numPCI] = -1;
				}
				else {
					printk(KERN_INFO NAME": Board[%d] set irq %d assigned OK!\n", numPCI, irq[numPCI]);

					// enable edge triggered LINTi1
					intctrl = readl(base0[numPCI]+0x4c);
					intctrl |= 1 + (1<<1) + (1<<8);
					//intctrl = (1<<4);
					writew(intctrl,base0[numPCI]+0x4c);    
 
					// enable interrupt
					intctrl = readl(base0[numPCI]+0x4c);
					intctrl |= (1<<6);
					writew(intctrl,base0[numPCI]+0x4c);            
					intctrl = readl(base0[numPCI]+0x4c);
					//printk("intctrl = 0x%x\n", intctrl);
				}
			}
			//printk("HW_FRAME=%d\n",HW_FRAME);
			card_detail(numPCI);
	     		numPCI++;
		}
    
	}

	printk(NAME": Detected %d V12PCI boards on PCI bus\n",numPCI);
	//printk(KERN_INFO NAME": Detected %d V12PCI boards on PCI bus\n",numPCI);

	// set up bitrev lut
	for(i=0; i<ALAW_CODES; i++) {
		bitrev_lut[i] = i;
	}
	bitrev(bitrev_lut, ALAW_CODES);

	// Set up Party Conferencing stuff
	#ifdef PARTY
	for(x=0;x<PARTY_MAX;x++){
		parties[x]=0;
		for(y=0;y<PARTY_MAXMEM;y++){
			party_book[x][y].board=-1;
			party_book[x][y].port=-1;
		}
	}
	for(x=0;x<3;x++){
		for(y=0;y<12;y++){
			party_people[x][y].party=-1;
			party_people[x][y].member=-1;
		}
	}
	party_count=0;
	#endif
	#ifdef BRIDGING
	for(x=0;x<10;x++){
		for(y=0;y<12;y++){
			london[x][y].board=-1;
			london[x][y].port=-1;
		}
	}
	/*
	london[0][0].board=1;
	london[0][0].port=0;
	london[1][0].board=0;
	london[1][0].port=0;
	*/
	#endif


	return 0;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: isr()
	AUTHOR...: David Rowe
	DATE.....: 22/12/01

	Called in response to hardware interrupts approximately once every
	ms.

\*--------------------------------------------------------------------------*/

void isr(int irq, void *dev_id, struct pt_regs *regs) {
	short *pbuf;
	u32  intctrl;
	int  i, j, ret,x;
	int bd = *(int*)dev_id;
	#ifdef BRIDGING
	int startrx=-1;
	int bridgeBoard, bridgePort;
	#endif
        #ifdef PARTY
	int startprx;
	long blah[PARTY_MAX][SIZE_FIFO/64];
	#endif

	//#define TIMING
	#ifdef TIMING
	u32  ctrl;
	#endif

	// clear edge triggered LINTi1
	intctrl = readl(base0[bd]+0x4c);
	intctrl |= (1<<10);
	writew(intctrl,base0[bd]+0x4c);   

	#ifdef TIMING
	// monitor WDR node on CRO to determine time spend in ISR
	// WDR H
        ctrl = readl(base0[bd]+0x50);
        ctrl |= (1<<5);
        writew(ctrl,base0[bd]+0x50);   
	#endif

	// transfer samples from tx software fifo to tx hardware fifo ------

	pbuf = frametx[bd];
	ret = fifo_read(tx_fifo[bd], (char*)pbuf, (SIZE_FIFO/2)*sizeof(short));
	if (ret == FIFO_EMPTY) {
		pbuf = default_tx;
	}
	else {
		pbuf = frametx[bd];
	}
	
	// Multi-Party Conferencing
	//
        #ifdef PARTY

	if (party_count >0){
		// There are some parties...
		// Find the start frame of the fifo buffer
		startprx = -1;
		for(i=0; i<32; i++) {
			if (framerx[bd][i] & 0x100){
				startprx = i;
				i=32;
			}
		}
		if (startprx != -1) {
			// Clear the blah buffers
			for(i=0; i<SIZE_FIFO/2; i+=32) {
				for(x=0;x<10;x++){
					blah[x][i/32]=0;
				}
			}
			//  Get all the samples we need from the fifo
			// and add them up
			for(i=0; i<SIZE_FIFO/2; i+=32) {
				for(x=0;x<12;x++){
					if(party_people[bd][x].party > -1){
						blah[party_people[bd][x].party][i/32] += alaw2linear(framerx[bd][i+startprx+x]&0xff);
					}
				}
			}
			// And send them back out the TX buffer
			for(i=0; i<SIZE_FIFO/2; i+=32) {
				for(x=0;x<12;x++){
					if(party_people[bd][x].party > -1){
						blah[party_people[bd][x].party][i/32]-=(framerx[bd][i+startprx+x]&0xff);
						if (blah[party_people[bd][x].party][i/32] > 32767)  blah[party_people[bd][x].party][i/32]=32767; 
						if (blah[party_people[bd][x].party][i/32] < -32767)  blah[party_people[bd][x].party][i/32]=-32767; 
						//frametx[bd][i+12+x] =  linear2alaw(blah[party_people[bd][x].party][i/32]*((1/parties[party_people[bd][x].party])+.2));
						frametx[bd][i+12+x] =  linear2alaw(blah[party_people[bd][x].party][i/32]);
					}
				}
			}
		}
	}
	// dodgey hack before software bridging was implemented
	if(sbridge ==1){
		startprx = -1;
		for(i=0; i<32; i++) {
			if (framerx[bd][i] & 0x100){
				startprx = i;
				i=32;
			}
		}
		if (startprx != -1) {
			for(i=0; i<SIZE_FIFO/2; i+=32) {
				frametx[bd][i+12+1] = framerx[bd][i+startprx]&0xff;
				frametx[bd][i+12] = framerx[bd][i+startprx+1]&0xff;
			}
		}
	}
	#endif


	// software bridging between boards
	
	#ifdef BRIDGING
	if (sbridge_count > 0){
		for(x=0;x<12;x++){
			if((london[bd][x].board != -1)&&(london[bd][x].port != -1)){
				bridgeBoard=london[bd][x].board;
				bridgePort=london[bd][x].port;
				startrx=-1;
				for(i=0; i<32; i++) {
					if (framerx[bridgeBoard][i] & 0x100){
						startrx = i;
						i=32;
					}
				}
				if (startrx != -1) {
					for(i=0; i<SIZE_FIFO/2; i+=32) {
						frametx[bd][i+12+x] = framerx[bridgeBoard][i+startrx+bridgePort]&0xff;
					}
				}
			}
		}
	}
	#endif
	
	// Write to TX FIFO.  Note FS pulse inserted here rather than in user
	// mode to ensure continuity of FS pulses even if user mode too slow
	// to fill sw FIFO.  We assume data from user mode is FS aligned.

	for(i=0; i<SIZE_FIFO/2; i+=32)
       	{
	    pbuf[0] |= 0x100;
//		for(j=0; j<HW_FRAME; j++) 
		for(j=0; j<32; j++) 
		    memcpy_toio(base2[bd], pbuf+j, sizeof(short));
		pbuf +=32;
	}
	
	// transfer samples from hardware rx fifo to software rx fifo -------

	// read from RX FIFO
	
	pbuf = framerx[bd];       
	for(i=0; i<SIZE_FIFO/2; i+=32)
       	{
	    for(j=0; j<HW_FRAME ; j++) {
		// Copy sample from Codec to memory
		memcpy_fromio(pbuf+j, base2[bd]+1, sizeof(short));
	    }
	    pbuf +=32;
	}

	pbuf = framerx[bd];       
	fifo_write(rx_fifo[bd], (char*)pbuf, (SIZE_FIFO/2)*sizeof(short));

	
	
	// update hook and ring detector samples
	khook_sample(khook[bd], board_type);
	kring_sample(kring[bd], board_type);
	kldrop_sample(kldrop[bd], board_type);
	

        #ifdef TIMING
	// WDR low
        ctrl = readl(base0[bd]+0x50);
        ctrl &= ~(1<<5);
        writew(ctrl,base0[bd]+0x50);            
	#endif
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: cleanup_module()
	AUTHOR...: David Rowe
	DATE.....: 15/4/00

	Called by the O/S when the module is unloaded. Frees any memory
	mapped areas for the PCI cards.

	TODO: DR 11/9/01: This function gives an error message on vfree under 
	2.4 kernel, not sure why....

\*--------------------------------------------------------------------------*/

void cleanup_module(void) {
	int ret;
	int i;
	u32 intctrl;

	// Release SW fifo memory

	for(i=0; i<numPCI; i++) {
		intctrl = readl(base0[i]+0x4c);
		//printk("intctrl = 0x%x\n", intctrl);

		// disable interrupts
		intctrl = readl(base0[i]+0x4c);
		intctrl &= ~(1<<6);
		writew(intctrl,base0[i]+0x4c);   
         
		if (irq[i] != -1)
	       	{
		    free_irq(irq[i], &boardNum[i]);
		}
  
		fifo_destroy(tx_fifo[i]);
		fifo_destroy(rx_fifo[i]);

		// unmap any PCI regions
		vfree(base0[i]);    
		release_region(base1[i], SIZE_LCR);
		vfree(base2[i]);
		khook_close(khook[i]);
		kring_close(kring[i]);
		kldrop_close(kldrop[i]);
	}
    
	ret = unregister_chrdev(major, NAME);
	if (ret < 0)
		printk(KERN_WARNING "unregister_chrdev() failed!\n");

	printk(KERN_INFO "vpbhp: succesfully removed \n");
	printk("vpbhp: succesfully removed \n");
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: vpbhp_ioctl()
	AUTHOR...: David Rowe
	DATE.....: 15/4/00

	Called when a user mode ioctl is issued, used to perform all
	PCI commands based on ioctl command.

\*--------------------------------------------------------------------------*/

static int vpbhp_ioctl(struct inode *inode,
			  struct file *filp,
			  unsigned int cmd,
			  unsigned long arg)
{
  int        ret;
  VPBHP_DATA vpbhp_data;  // ioctl parameters from user space
  int        vpbhp_addr;  // addr in card memory
  int        length;      // length of transfer in words
  short      *data;       // ptr to user space data
  int        pci_num;     // pci card number
  int        ctrl;        // value of PLX9052 control reg
  int        i;
  short		x,y;
  short      *pbuf;       // source/dest for next block
  int        howfull;
  int        howempty;
  int        khook_rd;
  int        kring_rd;
  int        kldrop_rd;

  // copy parameter data from user address space

  ret = copy_from_user(&vpbhp_data, (void*)arg, sizeof(VPBHP_DATA));
  if (ret != 0)
  {
      printk(KERN_CRIT "verify area VPBHP_DATA failed\n");
      return -ENODEV;
  }

  // local copies of parameters

  vpbhp_addr = vpbhp_data.vpbhp_addr;
  if (vpbhp_addr > SIZE_WD) 
  {
      printk(KERN_WARNING "vpbhp_addr > SIZE_WD, ioctl failed!\n");    
      return -ENODEV;
  }
  length = vpbhp_data.length;
  if (length > SIZE_BUF)
  {
      printk(KERN_WARNING "length > SIZE_BUF, ioctl failed!\n");    
      return -ENODEV;
  }
  data = vpbhp_data.data;
  pci_num = vpbhp_data.pci_num;

  switch(cmd) {
    case VPBHP_IOC_PCI_SBRIDGE_ON:
	    printk(KERN_INFO NAME ": Bridging port %d on board %d to port %d on board %d\n",vpbhp_addr,pci_num,((int *)data)[1],((int *)data)[0]);
	    london[pci_num][vpbhp_addr].board=((int *)data)[0];
	    london[pci_num][vpbhp_addr].port=((int *)data)[1];
	    sbridge_count++;
    break;
    case VPBHP_IOC_PCI_SBRIDGE_OFF:
	    printk(KERN_INFO NAME ": UN-Bridging port %d on board %d to port %d on board %d\n",vpbhp_addr,pci_num,london[pci_num][vpbhp_addr].port,london[pci_num][vpbhp_addr].board);
	    london[pci_num][vpbhp_addr].board=-1;
	    london[pci_num][vpbhp_addr].port=-1;
	    sbridge_count--;
    break;
    case VPBHP_IOC_PCI_CONF_JOIN:
	if ((short)*data == -1){
		sbridge=1;
		break;
	}
	for(x=0;x<10;x++){
		if(party_book[(short )*data][x].board == -1){
			party_book[(short )*data][x].board=pci_num;
			party_book[(short )*data][x].port=vpbhp_addr;
			party_people[pci_num][vpbhp_addr].party=(int)*data;
			party_people[pci_num][vpbhp_addr].member=x;
			x=10;
			if (parties[(int)*data] == 0){
				party_count++;
			}
			parties[(int)*data]++;
		}
	}
    break;

    case VPBHP_IOC_PCI_CONF_LEAVE:
	y=party_people[pci_num][vpbhp_addr].member;
	x=party_people[pci_num][vpbhp_addr].party;
	party_book[x][y].board=-1;
	party_book[x][y].port=-1;
	party_people[pci_num][vpbhp_addr].party=-1;
	party_people[pci_num][vpbhp_addr].member=-1;
	parties[x]--;
	if (parties[x] == 0){
		party_count--;
	}
    break;


    case VPBHP_IOC_PCI_GET_NUM_CARDS:
      if (numPCI==0)
	return -ENODEV;
      copy_to_user(data, &numPCI, sizeof(int));
    break;

    case VPBHP_IOC_PCI_BLOCK_WRITE:
      if (numPCI==0)
	return -ENODEV;
      copy_from_user(buf, data, length*sizeof(short));

      // copy block of memory to single V12PCI card address

      pbuf = buf;
      for(i=0; i<length; i++) {
          memcpy_toio(base2[pci_num]+vpbhp_addr, pbuf++, sizeof(short));
      }
    break;
     
    case VPBHP_IOC_PCI_BLOCK_READ:
 
      if (numPCI==0)
	return -ENODEV;

      // copy block of memory from single V12PCI card address
      pbuf = buf;
      for(i=0; i<length; i++) {
        memcpy_fromio(pbuf++, base2[pci_num]+vpbhp_addr, sizeof(short));
      }
      copy_to_user(data, buf, length*sizeof(short));
    break;
    
    case VPBHP_IOC_PCI_WD_EN_L:
      if (numPCI==0)
	return -ENODEV;

      // reset PLX9052 pin user0, connected to WD enable, disabling WD timer
      ctrl = readl(base0[pci_num]+0x50);
      ctrl &= ~(1<<2);
      writew(ctrl,base0[pci_num]+0x50);            
    break;
    
    case VPBHP_IOC_PCI_WD_EN_H:
      if (numPCI==0)
	return -ENODEV;

      // set PLX9052 pin user0, connected to WD enable, enabling WD
      ctrl = readl(base0[pci_num]+0x50);
      ctrl |= (1<<2);
      writew(ctrl,base0[pci_num]+0x50);            
    break;
    
    case VPBHP_IOC_PCI_WDR_L:
      if (numPCI==0)
	return -ENODEV;

      // reset PLX9052 pin user1, connected to WD reset, de-asserting WD reset
      ctrl = readl(base0[pci_num]+0x50);
      ctrl &= ~(1<<5);
      writew(ctrl,base0[pci_num]+0x50);            
    break;
    
    case VPBHP_IOC_PCI_WDR_H:
      if (numPCI==0)
	return -ENODEV;

      // set PLX9052 pin user1, connected to WD reset, asserting WD
      // Note: WD reset is rising-edge triggered
      ctrl = readl(base0[pci_num]+0x50);
      ctrl |= (1<<5);
      writew(ctrl,base0[pci_num]+0x50);            
    break;
    
    case VPBHP_IOC_PCI_TX_FIFO_BLOCK_WRITE:
      if (numPCI==0)
	return -ENODEV;

      copy_from_user(buf, data, length*sizeof(short));
      fifo_write(tx_fifo[pci_num], (char*)buf, length*sizeof(short));
    break;

    case VPBHP_IOC_PCI_RX_FIFO_BLOCK_READ:
      if (numPCI==0)
	return -ENODEV;

      fifo_read(rx_fifo[pci_num], (char*)buf, length*sizeof(short));
      copy_to_user(data, buf, length*sizeof(short));
    break;

    case VPBHP_IOC_PCI_TX_FIFO_HOW_EMPTY:
      if (numPCI==0)
	return -ENODEV;
      fifo_how_full(tx_fifo[pci_num], &howfull);
      howempty = (SIZE_SW_FIFO - howfull/sizeof(short));
      copy_to_user(data, &howempty, sizeof(int));
    break;

    case VPBHP_IOC_PCI_RX_FIFO_HOW_FULL:
      if (numPCI==0)
	return -ENODEV;
      fifo_how_full(rx_fifo[pci_num], &howfull);
      howfull /= sizeof(short);
      copy_to_user(data, &howfull, sizeof(int));
    break;
	  
    case VPBHP_IOC_PCI_READ_KHOOK:
      if (numPCI==0)
	return -ENODEV;
      khook_rd = khook_read(khook[pci_num], vpbhp_addr);
      copy_to_user(data, &khook_rd, sizeof(int));
    break;
	  
    case VPBHP_IOC_PCI_READ_KRING:
      if (numPCI==0)
	return -ENODEV;
      kring_rd = kring_read(kring[pci_num], vpbhp_addr);
      copy_to_user(data, &kring_rd, sizeof(int));
    break;
     
    case VPBHP_IOC_PCI_BLOCK_EEREAD:
 
      if (numPCI==0)
	return -ENODEV;
      pbuf = buf;
      for(i=0; i<length; i++)
      {
        *(pbuf++) =eeread(pci_num,vpbhp_addr+i);
      }
      copy_to_user(data, buf, length*sizeof(short));
    break;

    case VPBHP_IOC_PCI_READ_KLDROP:
      if (numPCI==0)
	return -ENODEV;
      kldrop_rd = kldrop_read(kldrop[pci_num], vpbhp_addr);
      copy_to_user(data, &kldrop_rd, sizeof(int));
    break;
  }
    
   return 0;
}

// user defined functions required by FIFO module

void *fifo_malloc(int size) {
	return vmalloc(size);
}

void fifo_free(void *mem) {
	vfree(mem);
}

void fifo_memcpy(char *dest, char *src, int length) {
  int i;
  for(i=0; i<length; i++) {
    *dest++ = *src++;
  }
}

// user defined functions required by khook module

void *khook_malloc(int size) {
	return vmalloc(size);
}

void khook_free(void *mem) {
	vfree(mem);
}

int khook_readw(unsigned short *addr) {
	short hk;
        memcpy_fromio(&hk, addr, sizeof(short));
	return hk;
}

// user defined functions required by kring module

void *kring_malloc(int size) {
	return vmalloc(size);
}

void kring_free(void *mem) {
	vfree(mem);
}

int kring_readw(unsigned short *addr) {
	short hk;
        memcpy_fromio(&hk, addr, sizeof(short));
	return hk;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: eeread(int addr)
	AUTHOR...: David Rowe 	Updates:Peter Wintulich
	DATE.....: 5-FEB-2003

	Card Data Read function. Reads a single word from eeprom addr.
\*--------------------------------------------------------------------------*/

unsigned int eeread(int PCIn, int addr) {
	int i;
	int d;

	cntrlbit(PCIn,EEPROM_CS,1);
	cntrlbit(PCIn,EEPROM_CLK,0);

	wrbit(PCIn,1); wrbit(PCIn,1); wrbit(PCIn,0);
	for(i=0; i<6; i++)
	    wrbit(PCIn,addr>>(5-i));
	
	d = 0;
	for(i=0; i<16; i++)
       	{
	    d <<= 1;
	    d += rdbit(PCIn);
	}
        cntrlbit(PCIn,EEPROM_CS,0);
        return(d);
}
                            
/*--------------------------------------------------------------------------*\

	FUNCTION.: cntrlbit(int PCInn, int bit, int val)
	AUTHOR...: David Rowe
	DATE.....: 5-FEB-2003

	Bit # set to val, Support function for eeprom read
\*--------------------------------------------------------------------------*/

void cntrlbit(int PCInn, int bit, int val)
{
	unsigned int cntrl;

	val &= 1;
     	// first reset bit
	cntrl = readl(base0[PCInn]+0x50);
	cntrl &= ~(1<<bit);
	// now set to val
	cntrl |= val<<bit;
	writel(cntrl,base0[PCInn]+0x50);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: wrbit(int PCInn, int bit)
	AUTHOR...: David Rowe
	DATE.....: 5-FEB-2003

	Write data bit, Support function for eeprom read
\*--------------------------------------------------------------------------*/

void wrbit(int PCInn, int bit) 
{
        cntrlbit(PCInn,EEPROM_WRBIT, bit);
        cntrlbit(PCInn,EEPROM_CLK, 1);
        cntrlbit(PCInn,EEPROM_CLK, 0);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: rdbit(int PCInn)
	AUTHOR...: David Rowe
	DATE.....: 5-FEB-2003

	Read data bit, Support function for eeprom read
\*--------------------------------------------------------------------------*/

int rdbit(int PCInn) 
{
        unsigned int cntrl;

        cntrlbit(PCInn,EEPROM_CLK, 1);
        cntrlbit(PCInn,EEPROM_CLK, 0);
        cntrl = readl(base0[PCInn]+0x50);
        cntrl >>= EEPROM_RDBIT;
        cntrl &= 1;
        return(cntrl);
}
                               
/*--------------------------------------------------------------------------*\

	FUNCTION.: card_detail(int pci_n)
	AUTHOR...: Peter Wintulich
	DATE.....: 5-FEB-2003

	Print details of card to console & log
\*--------------------------------------------------------------------------*/

void card_detail(int pci_n)
{
	int 	i, x=0;
	char	st[16];
	char	hex[17]={"0123456789abcdef"};
	unsigned short eebuf[20];
	
	i=50;
	while(i<56)
	    eebuf[x++] = eeread(pci_n,i++);	

	// Decode date
	x=i=0;
	st[x++]=hex[((eebuf[i]>>12)& 0x0f)];	//day
	st[x++]=hex[((eebuf[i]>>8)& 0x0f)];
	st[x++]='/';
	st[x++]=hex[((eebuf[i]>>4)& 0x0f)];	//month
	st[x++]=hex[(eebuf[i] & 0x0f)];
	st[x++]='/';
	i++;
	st[x++]=hex[((eebuf[i]>>12)& 0x0f)];	//year
	st[x++]=hex[((eebuf[i]>>8)& 0x0f)];
	st[x++]=hex[((eebuf[i]>>4)& 0x0f)];
	st[x++]=hex[(eebuf[i] & 0x0f)];
	st[x]=0;	
	i++;
	printk(NAME ": Manufactured %s\n",st);
	
	x=0;
	st[x++]=hex[((eebuf[i]>>12)& 0x0f)];	//card
	st[x++]=hex[((eebuf[i]>>8)& 0x0f)];
	st[x++]=0;
	printk(NAME ": Card version %s.",st);

	x=0;
	st[x++]=hex[((eebuf[i]>>4)& 0x0f)];	//Rev.
	st[x++]=hex[(eebuf[i] & 0x0f)];
	st[x++]=0;
	i++;
	printk("%s\n",st);

	i++;					// Resurved word;
	
	x=0;
	st[x++]=hex[((eebuf[i]>>12)& 0x0f)];	//SN:high
	st[x++]=hex[((eebuf[i]>>8)& 0x0f)];
	st[x++]=hex[((eebuf[i]>>4)& 0x0f)];
	st[x++]=hex[(eebuf[i] & 0x0f)];
	i++;
	st[x++]=hex[((eebuf[i]>>12)& 0x0f)];	//SN:low
	st[x++]=hex[((eebuf[i]>>8)& 0x0f)];
	st[x++]=hex[((eebuf[i]>>4)& 0x0f)];
	st[x++]=hex[(eebuf[i] & 0x0f)];
	st[x]=0;
	printk(NAME ": Serial number %s\n",st);
}	

// user defined functions required by kldrop module

void *kldrop_malloc(int size) {
	return vmalloc(size);
}

void kldrop_free(void *mem) {
	vfree(mem);
}

int kldrop_readw(unsigned short *addr) {
	short hk;
        memcpy_fromio(&hk, addr, sizeof(short));
	return hk;
}

int alaw2linear( short alawbyte) 
{
	    unsigned long  acc,p,s,q;
	    int    sample;

	alawbyte = bitrev_lut[alawbyte];

	acc = alawbyte ^ 0x55;   /* even bit inversion */

	/* extract sign */
	if (acc & 0x80)
		p = 0x8000;
	else
		p = 0;

	acc &= 0x7f;

	/* extract q and s */
	q = acc & 0xf;
	s = acc >> 4;

	/* form linear sample */
	if (s != 0) {
		acc = 2*q+33;
		acc <<= (s-1);
	}
	else
		acc = 2*q+1;

	acc <<= 3;
	if (p)
	sample = -(short)acc;
	else
	sample = (short)acc;

	return(sample);
}

short linear2alaw(int sample)
{
	unsigned short y,mag;
	unsigned short p,s,q,b;
	unsigned long  acc;
	short alawbyte;
	acc = (sample >> 3) & 0xffff;

	/* separate sign */
	if (acc & 0x8000) {
		p = 0x80;
		mag = -(short)acc;
	}
	else {
		p = 0;
		mag = (short)acc;
	}

	/* encode magnitude */
	if (mag > 0x0fff) 
	{
		y = p + 0x7f;
	}
	else {
		/* determine left most bit and therefore segment */
		b = 0;
		acc = mag >> 1;
		while(acc) {
			b++;
			acc >>= 1;
		}

		if (b > 4) {
			s = b - 4;
			q = mag >> (b-4);
			q &= 0xf;
		}
		else {
			s = 0;
			q = mag >> 1;
		}

		y = p + (s<<4) + q;
	}

	alawbyte= y^0x55;               /* even bit inversion */

	alawbyte = bitrev_lut[alawbyte];
	return(alawbyte);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: bitrev
	AUTHOR...: David Rowe
	DATE.....: 5/6/02

	Bit reversal of samples - hardware bug work around.

\*--------------------------------------------------------------------------*/

void bitrev(unsigned char a[], int n) {
  int i,j;
  unsigned char b,c;
 
  for(i=0; i<n; i++) {
    b = a[i];
    c = 0;
    for(j=0; j<8; j++) {
      c |= ((b>>j)&0x1) << (7-j);
    }
    a[i] = c;
  }
}
